# Spreadsheet tag importer UI scripts and import algorithms.

# Copyright 2023-2024 Automation Professionals, LLC <sales@automation-pros.com>
#
# Redistribution and use in source and binary forms, with or without modification,
# are permitted provided that the following conditions are met:
#
#   1. Redistributions of source code must retain the above copyright notice,
#      this list of conditions and the following disclaimer.
#   2. Redistributions in binary form must reproduce the above copyright notice,
#      this list of conditions and the following disclaimer in the documentation
#      and/or other materials provided with the distribution.
#   3. The name of the author may not be used to endorse or promote products
#      derived from this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED
# WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
# SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
# EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
# OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
# CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
# IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
# OF SUCH DAMAGE.

from java.lang import System, Throwable, Double, Long, Integer, RuntimeException, StringBuilder
from java.io import File, FileInputStream, ByteArrayInputStream
from java.util import Date
from java.util.concurrent.atomic import AtomicInteger
from java.time import Instant, ZoneId
from java.time.format import DateTimeFormatter
from org.apache.poi.xwpf.usermodel import XWPFDocument, XWPFParagraph, XWPFRun, XWPFTable, XWPFTableRow, XWPFTableCell
from org.apache.poi.ss.usermodel import WorkbookFactory, DateUtil, DataFormatter
from org.apache.poi.util import TempFile, DefaultTempFileCreationStrategy

from com.inductiveautomation.ignition.common import TypeUtilities
from com.inductiveautomation.ignition.common.sqltags.model.types import DataType
from com.inductiveautomation.ignition.common.tags.paths.parser import TagPathParser
from com.inductiveautomation.ignition.common.tags.config.properties import ParameterValue, Parameter, WellKnownTagProps
import traceback

from collections import OrderedDict as oDict

logger = system.util.getLogger(system.util.getProjectName()+'.'+system.reflect.getModulePath())

# Ensure temporary files can be created by Apache POI.
ctx = system.util.toolkitCtx()
try:
	TempFile.setTempFileCreationStrategy(DefaultTempFileCreationStrategy(ctx.systemManager.tempDir))
except AttributeError:
	TempFile.setTempFileCreationStrategy(DefaultTempFileCreationStrategy(ctx.launchContext.gwCacheDir))

# Intelligent String formatting of timestamps, with milliseconds.
defaultFormatter = DateTimeFormatter.ofPattern("YYYY-MM-dd HH:mm:ss.SSS")
def zonedFormat(ts, zoneid, format=None):
	if not isinstance(ts, Instant):
		ts = ts.toInstant()
	formatter = defaultFormatter if format is None else DateTimeFormatter.ofPattern(format)
	return ts.atZone(ZoneId.of(zoneid)).format(formatter)

# Establish some introspection values
gvm = system.util.globalVarMap('tagImporter')
def initProviders():
	gvm['providers'] = [x['name'] for x in system.tag.browse("", {})]
	gvm.refresh()

# Reset Progress in pageVarMap on settings change
def resetProgress():
	pvm = system.perspective.pageVarMap()
	pvm.tagImportProgress = ''
	pvm.tagImportBusy = False
	pvm.refresh()

# Configuration object for provider name and forced creation settings
# An instance of this will be created in Perspective scope then handed to
# the asynchronous task that does the work.  It will be carried as a
# mutable function argument into all function levels, recursively.
class UploadConfig(object):
	def __init__(self, this):
		custom = this.view.custom
		self.pvm = system.perspective.pageVarMap()
		self.provider = this.view.custom.provider
		self.__dict__.update(this.view.custom.options)
		deviceDS = system.device.listDevices()
		self.devices = set(deviceDS.getColumnAsList(deviceDS.getColumnIndex('Name')))
		self.servers = set(system.opc.getServers(True))
		self.devicesCreated = set()
		self.serversCreated = set()
		self.udtsCreated = set()
		self.udtsReported = {}
		self.tagsCreated = set()
		self.tagsToDelete = oDict()
		self.tzId = this.session.props.timeZoneId

# Handle a file upload in the Spreadsheet Import page.
# Create the configuration object, then delegate to a background task.
def upload(self, event):
	# Note that "self" here is "this" inside the UploadConfig constructor.
	# The container nesting of the view is ignored--all user settings are
	# bidirectionally bound to view custom properties for script access.
	config = UploadConfig(self)
	system.util.invokeAsynchronous(processUpload, [self, event, config], description='Spreadsheet tag import')

startTemplate = """## %s Processing %s %d bytes => Provider %s\n"""

# Function for use with system.util.invokeAsynchronous() to do the bulk of the work.
# Note that "view" message scope cannot be targeted from an asynchronous thread.
def processUpload(self, event, config):
	# The progress markdown component is bound via pageVarMap() to the following
	# property, but the binding triggering will be paced.
	config.pvm.tagImportProgress = startTemplate % (zonedFormat(Date(), config.tzId), event.file.name, event.file.size, config.provider)
	config.pvm.tagImportBusy = True
	config.refreshTS = System.nanoTime()
	config.pvm.refresh()

	# Throttle refreshes on the progress display.
	def progress(txt):
		config.pvm.tagImportProgress += "\n\t"
		lines = txt.splitlines()
		if lines:
			config.pvm.tagImportProgress += "\n\t".join(lines)
		ts = System.nanoTime()
		# Progress updates no faster than 1/4 second.
		if ts - config.refreshTS > 250000000L:
			config.refreshTS = ts
			config.pvm.refresh()

	try:
		baos = ByteArrayInputStream(event.file.bytes)
		processWorkbook(WorkbookFactory.create(baos), progress, config)
	except Throwable, t:
		logger.warn("Unexpected java error handling "+event.file.name, t)
		config.pvm.tagImportProgress += "\n\tEarly termination due to java error: "+t.message+"\n\tCheck your gateway log for details."
	except Exception, e:
		t = exchange.spreadsheetImportTool.later.PythonAsJavaException(e)
		logger.warn("Unexpected python error handling "+event.file.name, t)
		config.pvm.tagImportProgress += "\n\tEarly termination due to python error: "+t.message+"\n\tCheck your gateway log for details."
	finally:
		config.pvm['tagImportBusy'] = False
		config.pvm.refresh()
		self.clearUploads()

# Outermost process.  Report sheet names then hand off to the sheet processor.
def processWorkbook(wb, progress, config):
	for sheet in wb:
		config.pvm.tagImportProgress += "\n## Sheet '%s'" % sheet.sheetName
		processSheet(sheet, progress, config)
		progress("End of Sheet '%s'\n" % sheet.sheetName)
	deletions = config.tagsToDelete.keys()
	if deletions:
		system.tag.deleteTags(deletions)
	config.pvm.tagImportProgress += "\n## %s End of Workbook" % zonedFormat(Date(), config.tzId)

def cellString(cell):
	if cell is None:
		return ""
	return DataFormatter().formatCellValue(cell)

def colNameForIdx(idx):
	c = idx + 1
	colName = ''
	while c:
		q, m = divmod(c - 1, 26)
		colName = chr(m+65) + colName
		c = q
	return colName

def cellSource(cell):
	colName = colNameForIdx(cell.columnIndex)
	return "%s:%s%d" % (cell.sheet.sheetName, colName, cell.rowIndex+1)

def cellValue(cell, lookup):
	if cell is None:
		return None
	try:
		n = cell.numericCellValue
		try:
			if n == double(int(n)):
				return int(n)
		except:
			pass
		try:
			if n == double(long(n)):
				return long(n)
		except:
			pass
		return n
	except:
		pass
	try:
		s = cell.stringCellValue
		s = s % lookup
		try:
			return Integer.decode(s)
		except:
			pass
		try:
			return Long.decode(s)
		except:
			pass
		try:
			return Double.parseDouble(s)
		except:
			pass
		return s
	except KeyError, k:
		t = exchange.spreadsheetImportTool.later.PythonAsJavaException(k)
		raise RuntimeException("cellValue() KeyError @ "+cellSource(cell)+" using "+repr(lookup), t)
	except Throwable, t:
		raise RuntimeException("cellValue() Error @ "+cellSource(cell), t)
	except Exception, e:
		t = exchange.spreadsheetImportTool.later.PythonAsJavaException(e)
		raise RuntimeException("cellValue() Error @ "+cellSource(cell), t)

def cellStringValue(cell, lookup):
	pattern = cellString(cell)
	try:
		return pattern % lookup
	except KeyError, k:
		t = exchange.spreadsheetImportTool.later.PythonAsJavaException(k)
		raise RuntimeException("cellStringValue() KeyError @ "+cellSource(cell)+" using "+repr(lookup), t)
	except Throwable, t:
		raise RuntimeException("cellStringValue() Error @ "+cellSource(cell), t)
	except Exception, e:
		t = exchange.spreadsheetImportTool.later.PythonAsJavaException(e)
		raise RuntimeException("cellStringValue() Error @ "+cellSource(cell), t)

headerTriggers = [
	{'Device', 'DeviceType', 'DevPropKey', 'DevPropValue'},
	{'OpcCx', 'OpcPropKey', 'OpcPropValue'},
	{'UdtTypePath', 'UdtInstancePath'},
	{'TagInstancePath'}
]

# Per-sheet process.  Scans for the column headers, then groups rows by device name.
def processSheet(sheet, progress, config):
	headerPairs = None
	blockRows = []
	emptyRowA = 0
	for row in sheet:
		s = cellString(row.getCell(0))
		if s and headerPairs and blockRows and row.rowNum > emptyRowA:
			# When a value shows up in column A under the above conditions,
			# a new block of rows is starting.  Process that block before
			# accumulating the rows of the next block.
			processBlock(headerPairs, blockRows, progress, config)
			del blockRows[:]
		if s and not headerPairs:
			candidateHeaders = [x for x in [(cell.columnIndex, cellString(cell).strip()) for cell in row] if x[1]]
			headerSet = set([h[1] for h in candidateHeaders])
			if any([headerSet >= triggerSet for triggerSet in headerTriggers]):
				headerPairs = candidateHeaders
				if config.verbose:
					progress("Column Headings: " + repr(candidateHeaders))
				continue
		if headerPairs:
			if s:
				emptyRowA = row.rowNum + 1
				blockRows.append(row)
			else:
				if blockRows:
					blockRows.append(row)
	# Handle the last block on the sheet.
	if headerPairs and blockRows:
		processBlock(headerPairs, blockRows, progress, config)

# Per-block process.  Start with a new master map and empty action list.
# Then perform columnwise iteration, and recurse for
# columns after an iterator column header.
def processBlock(headerPairs, blockRows, progress, config):
	masterMap = {'_actionList': []}
	processBlockColumns(masterMap, headerPairs, blockRows, progress, config, 0, "")

# Recursive column process.
def processBlockColumns(outerMap, headerPairs, blockRows, progress, config, columnIdx, prefix, iteratorRow=None):
	# Shallow copy, so the action list is a shared list.
	# This captures the dictionary state from prior columns
	currentMap = dict(outerMap)

	# Now, iterate from the current column and to the right.
	for idx, (col, header) in list(enumerate(headerPairs))[columnIdx:]:
		# Header Pairs that end in "Key" and "Value" (in that order), with
		# the same prefix, indicate the construction of a nested dictionary.
		# The values inserted into the nested dictionary will also be
		# inserted into the current dictionary, and the nested dictionary
		# itself, if the prefix is associated with an action, will also be
		# inserted into the current dictionary, using the prefix as the key.
		if header.endswith('Value'):
			priorHeader = headerPairs[idx-1][1]
			if not priorHeader.endswith('Key') or header[:-5] != priorHeader[:-3]:
				# This can only occur if the prior column header didn't
				# end with "Key".
				raise ValueError('Column ' + header + ' does not follow column ' + header[:-5] + 'Key')
			continue
		if header.endswith('Key'):
			nextCol, nextHeader = headerPairs[idx+1]
			if nextHeader != header[:-3] + 'Value':
				raise ValueError('Column %s is followed by %s @ %d, requires %s' % (header, nextHeader, nextCol, header[:-3] + 'Value'))
			# Check for special names that will used by actions
			if header in ('DevPropKey', 'OpcPropKey', 'UdtParamKey', 'UdtPropKey', 'UdtMemberKey', 'TagPropKey'):
				actionMap = {}
				currentMap[header] = actionMap
			else:
				actionMap = None
			for row in blockRows:
				mapKey = cellString(row.getCell(col))
				mapValue = cellValue(row.getCell(nextCol), currentMap)
				if mapKey and mapValue is not None:
					currentMap[mapKey] = mapValue
					# Action maps get copies of these values
					if actionMap is not None:
						actionMap[mapKey] = mapValue
			continue

		# Iterator columns are a boundary that causes actions to their
		# left to be executed with the information in the current dictionary
		# so far, with recursion into the right hand columns.  After each
		# iteration, the dictionary built up to the right is discarded before
		# starting anew.
		if header.endswith('Iterator'):
			mapKey = header[:-8]
			if not mapKey:
				raise ValueError("'Iterator' may not be a column name by itself")
			# Execute and remove actions that were declared before this iterator.
			actions = currentMap['_actionList']
			while actions:
				action = actions.pop(0)
				action(currentMap, progress, config, prefix)
			# Iterators may have integers and integer ranges on any row in the device block.
			# printf-style substitutions from the map are allowed before the integer or
			# integers are parsed.
			progress(prefix+"Looping for %s" % header)
			for row in blockRows:
				iterCell = row.getCell(col)
				iterValues = cellStringValue(iterCell, currentMap).strip()
				if iterValues:
					ranges = iterValues.split(',')
					for rng in ranges:
						pair = rng.split('-')
						if (len(pair) == 2):
							try:
								start = Integer.decode(pair[0].strip())
								finish = Integer.decode(pair[1].strip())
							except Throwable, t:
								raise RuntimeException("Failed to parse iterator range %s in %s" % (rng, cellSource(iterCell)), t)
							for solo in range(start, finish+1):
								processIterator(mapKey, solo, currentMap, headerPairs, blockRows, progress, config, idx+1, prefix+"  ", row)
						else:
							rng = rng.strip()
							if rng:
								try:
									solo = Integer.decode(rng)
								except Throwable, t:
									raise RuntimeException("Failed to parse iterator solo %s in %s" % (rng, cellSource(iterCell)), t)
								processIterator(mapKey, solo, currentMap, headerPairs, blockRows, progress, config, idx+1, prefix+"  ", row)
			return
		if header.endswith('Eval'):
			mapKey = header[:-4]
			if not mapKey:
				raise ValueError("'Eval' may not be a column name by itself")
			if iteratorRow:
				expression = cellString(iteratorRow.getCell(col))
				if not expression:
					expression = cellString(blockRows[0].getCell(col))
			else:
				expression = cellString(blockRows[0].getCell(col))
			currentMap[mapKey] = eval(expression, dict(currentMap))
#			progress(prefix+"  Eval %s => %s" % (expression, currentMap[mapKey]))
		else:
			mapKey = header
			if not mapKey:
				continue
			if iteratorRow and cellString(iteratorRow.getCell(col)):
				currentMap[mapKey] = cellValue(iteratorRow.getCell(col), currentMap)
			else:
				currentMap[mapKey] = cellValue(blockRows[0].getCell(col), currentMap)
#			progress(prefix+"  Subst => %s" % currentMap[mapKey])
		if mapKey in ('DeviceType', 'OpcCx', 'UdtTypePath', 'TagInstancePath'):
			actions = currentMap['_actionList']
			actions.append(globals()['action'+mapKey])

	# Normal End of columns.  Perform any remaining actions.
	actions = currentMap['_actionList']
	while actions:
		action = actions.pop(0)
		action(currentMap, progress, config, prefix)
	# End of recursive process

# Repeating lines reduction.
# When iterating in columns, any ordinary columns (not ending in Key, Value, or
# Iterator, also rows ending in Eval) may have content in any additional rows
# of the block, and will use the content in the same row as the current iterator
# value.  (Unless blank, where the column content in the first row of the block
# will be used.)
def processIterator(key, val, currentMap, headerPairs, blockRows, progress, config, idx, prefix, iteratorRow):
	currentMap[key] = val
	processBlockColumns(currentMap, headerPairs, blockRows, progress, config, idx, prefix, iteratorRow)

# Device creation action.
# Uses action map stored under key 'DevPropKey'.
def actionDeviceType(currentMap, progress, config, prefix):
	if config.deviceMode > 2:
		# Skipping all device operations
		return
	propsMap = currentMap.pop('DevPropKey', {})
	description = propsMap.pop('description', None)
	device = currentMap.get('Device')

	kwargs = oDict(deviceName = device, deviceType = currentMap.get('DeviceType'), deviceProps = propsMap)
	if description:
		kwargs['description'] = description

	if device in config.devices:
		if config.deviceMode == 2:
			# Delete existing device
			progress(prefix+"Deleting Device %s" % device)
			config.devices.discard(device)
			try:
				system.device.removeDevice(device)
			except:
				progress(prefix+"Failed to delete Device %s\n%s" % (device, traceback.format_exc()))
			return
		if config.deviceMode and not device in config.devicesCreated:
			if config.verbose:
				progress(prefix+"Replacing Device %s" % repr(kwargs))
			else:
				progress(prefix+"Replacing Device %s" % device)
			system.device.removeDevice(device)
		else:
			progress(prefix+"Skipping Device %s Creation" % device)
			return
	else:
		if config.deviceMode == 2:
			# Nothing to delete
			return
		if config.verbose:
			progress(prefix+"Creating Device %s" % repr(kwargs))
		else:
			progress(prefix+"Creating Device %s" % device)
	config.devicesCreated.add(device)

	# Note:  Modern system.device.addDevice() **REQUIRES** keyword arguments.
	try:
		system.device.addDevice(**kwargs)
	except:
		progress(prefix+"Failed to create Device %s\n%s" % (repr(kwargs), traceback.format_exc()))
#

# OPC UA connection creation action.
# Uses action map stored under key 'OpcPropKey'.

# Certain keys in that map are extracted for top-level arguments to
# system.opcua.addConnection().  The rest are delivered to the settings
# argument.
opcArgs = ['description', 'discoveryUrl', 'endpointUrl', 'securityPolicy', 'securityMode']
#
def actionOpcCx(currentMap, progress, config, prefix):
	if config.opcMode > 2:
		# Skipping all OPC UA connection operations
		return
	propsMap = currentMap.pop('OpcPropKey', {})
	cxName = currentMap.get('OpcCx')
	args = [cxName] + [propsMap.pop(k, None) for k in opcArgs]
	args += [propsMap]
	kwargs = oDict(zip(['name'] + opcArgs + ['settings'], args))

	if cxName in config.servers:
		if config.opcMode == 2:
			# Delete existing connection
			progress(prefix+"Deleting OPC Connection %s" % cxName)
			config.servers.discard(cxName)
			try:
				system.opcua.removeConnection(cxName)
			except:
				progress(prefix+"Failed to delete OPC Connection %s\n%s" % (cxName, traceback.format_exc()))
			return
		if config.opcMode and not cxName in config.serversCreated:
			if config.verbose:
				progress(prefix+"Replacing OPC Connection %s" % repr(kwargs))
			else:
				progress(prefix+"Replacing OPC Connection %s" % cxName)
			system.opcua.removeConnection(cxName)
		else:
			progress(prefix+"Skipping OPC Connection %s Creation" % cxName)
			return
	else:
		if config.opcMode == 2:
			# Nothing to delete
			return
		if config.verbose:
			progress(prefix+"Creating OPC Connection %s" % repr(kwargs))
		else:
			progress(prefix+"Creating OPC Connection %s" % cxName)
	config.serversCreated.add(cxName)

	try:
		system.opcua.addConnection(*args)
	except:
		progress(prefix+"Failed to create OPC Connection %s\n%s" % (repr(kwargs), traceback.format_exc()))

# UDT Instance creation action.
# Uses action maps stored under keys 'UdtParamKey', 'UdtPropKey', and 'UdtMembersKey'.
def actionUdtTypePath(currentMap, progress, config, prefix):
	if config.udtMode > 2:
		# Skipping all UDT Instance operations
		return
	instancePath = currentMap.get('UdtInstancePath')
	tp = TagPathParser.parse("[%s]%s" % (config.provider, instancePath))
	tpFull = tp.toStringFull()
	if config.udtMode == 2:
		if tpFull in config.tagsToDelete:
			return
		progress(prefix + "Deleting " + instancePath)
		config.tagsToDelete[tpFull] = True
		return
	typePath = currentMap.get('UdtTypePath')
	paramsMap0 = currentMap.pop('UdtParamKey', {})
	propsMap = currentMap.pop('UdtPropKey', {})
	membersMap = currentMap.pop('UdtMemberKey', {})

	# Prepare to convert parameter values to nested parameter type and value.
	# Retrieve the parameter definitions from the UDT to get their expected types.
	udt = config.udtsReported.get(typePath)
	if udt is None:
		utp = TagPathParser.parse("[%s]_types_/%s" % (config.provider, typePath))
		udt = system.tag.getConfiguration(utp, True)
		if udt:
			udt = udt[0]
		else:
			progress(prefix+"Failed to retrieve UDT Type %s for Instance %s" % (typePath, instancePath))
			return
		config.udtsReported[typePath] = udt
		if config.verbose:
			progress(prefix+" UDT Type %s definition: %s" % (typePath, repr(udt)))
	# Parameters misbehave if included in system.tag.configure.  We will use .writeBlocking afterwards.
	# The same writeBlocking call will handle member writes and member property writes.

	propsMap['name'] = tp.lastPathComponent
	propsMap['tagType'] = 'UdtInstance'
	propsMap['typeId'] = typePath
	pp = tp.parentPath
	if config.verbose:
		progressText = " UDT Instance %s Type %s with parameters %s, properties %s, and member overrides %s" % (
			instancePath, typePath, repr(paramsMap0), repr(propsMap), repr(membersMap))
	else:
		progressText = " UDT Instance %s Type %s" % (instancePath, typePath)
	if system.tag.exists(tpFull):
		if config.udtMode and not instancePath in config.udtsCreated:
			policy = 'o'
			progress(prefix + "Overwriting" + progressText)
		else:
			policy = 'm'
			progress(prefix + "Merging" + progressText)
	else:
		policy = 'o'
		progress(prefix + "Creating" + progressText)
	config.udtsCreated.add(instancePath)
#
	paramDefs = udt.get('parameters', {})
	paramsPaths = []
	paramsValues = []
	paramsp = tp.getChildPath('Parameters')
	for k, v in paramsMap0.items():
		if k in paramDefs:	
			pv = ParameterValue(paramDefs[k].datatype, v)
			paramsPaths.append(paramsp.getChildPath(k).toStringFull())
			paramsValues.append(pv.value)
#
	for k, v in membersMap.items():
		mp = TagPathParser.parse(k)
		if mp.source or not mp.pathLength:
			progress(prefix+" Invalid Member Key %s for UDT Instance %s" % (k, instancePath))
			return
		udtPosition = udt
		mcp = None
		for ci in range(mp.pathLength):
			cname = mp.getPathComponent(ci)
			innerTags = udtPosition.get('tags', [])
			for mdef in innerTags:
				if mdef['name'] == cname:
					udtPosition = mdef
					mcp = (tp if mcp is None else mcp).getChildPath(cname)
					break
			else:
				progress(prefix+" Unknown Member Key %s for UDT Instance %s" % (k, instancePath))
				return
		if mp.property:
			mcp = mcp.getChildPath(mp.property)
		paramsPaths.append(mcp.toStringFull())
		paramsValues.append(v)
		if WellKnownTagProps.Value.equals(mcp.property) or not mcp.property:
			mtype = udtPosition.get(WellKnownTagProps.DataType.name)
			if DataType.Document.equals(mtype):
				# Spreadsheet values for document tags are expected to be JSON encoded.
				# Decode for assignment
				paramsValues[-1] = system.util.jsonDecode(v)
			elif DataType.DataSet.equals(mtype):
				paramsValues[-1] = system.dataset.fromCSV(v)
		elif WellKnownTagProps.Alarms.equals(mcp.property):
			paramsValues[-1] = system.util.jsonDecode(v)
#
	try:
		rc = system.tag.configure(pp.toStringFull(), [propsMap], policy)
		if not rc[0].good:
			progress(prefix+" Non-good result: "+str(rc[0]))
		else:
			if paramsPaths:
				system.tag.writeBlocking(paramsPaths, paramsValues)
	except:
		progress(prefix+" Failure:\n\t"+traceback.format_exc())#

# Solo Tag creation action.
# Uses action map stored under key 'TagPropKey'.
def actionTagInstancePath(currentMap, progress, config, prefix):
	if config.tagMode > 2:
		# Skipping all Atomic Tag actions
		return
	instancePath = currentMap.get('TagInstancePath')
	tp = TagPathParser.parse("[%s]%s" % (config.provider, instancePath))
	tpFull = tp.toStringFull()
	if config.tagMode == 2:
		if tpFull in config.tagsToDelete:
			return
		progress(prefix + "Deleting " + instancePath)
		config.tagsToDelete[tpFull] = True
		return

	propsMap = currentMap.pop('TagPropKey', {})
	propsMap['name'] = tp.lastPathComponent
	newTagValue = propsMap.pop('value', None)
	if 'alarms' in propsMap:
		propsMap['alarms'] = system.util.jsonDecode(propsMap['alarms'])
	pp = tp.parentPath
	if config.verbose:
		progressText = " Tag Instance %s with properties %s" % (instancePath, repr(propsMap))
	else:
		progressText = " Tag Instance %s" % instancePath
	if system.tag.exists(tpFull):
		if config.tagMode and not instancePath in config.tagsCreated:
			policy = 'o'
			progress(prefix + "Overwriting" + progressText)
		else:
			policy = 'm'
			progress(prefix + "Merging" + progressText)
	else:
		policy = 'o'
		progress(prefix + "Creating" + progressText)
	config.tagsCreated.add(instancePath)

	try:
		dtype = DataType.valueOf(propsMap.setdefault('dataType', 'Document'))
		if not newTagValue is None:
			if dtype is DataType.Document:
				subst = system.util.jsonDecode(newTagValue)
			elif dtype is DataType.DataSet:
				subst = system.dataset.fromCSV(newTagValue)
			else:
				subst = TypeUtilities.coerce(newTagValue, dtype.javaType)
			newTagValue = subst
	except:
		progress(prefix+"Failed to handle data type for %s\n%s" % (repr(propsMap), traceback.format_exc()))

	try:
		rc = system.tag.configure(pp.toStringFull(), [propsMap], policy)
		if not rc[0].good:
			progress(prefix+" Non-good configure() result: "+str(rc[0]))
			return
		if not newTagValue is None:
			qList = system.tag.writeBlocking([tpFull], [newTagValue])
			if not qList[0].good:
				progress(prefix+" Non-good value write: "+str(qList[0]))
	except:
		progress(prefix+" Failure:\n\t"+traceback.format_exc())

#